//////////////////////////////////////////////////////////////////////////////
//  Copyright 2014 by Autodesk, Inc.  All rights reserved.
//
// This computer source code and related instructions and comments 
// are the unpublished confidential and proprietary information of 
// Autodesk, Inc. and are protected under applicable copyright and 
// trade secret law.  They may not be disclosed to, copied or used 
// by any third party without the prior written consent of Autodesk, Inc.
//////////////////////////////////////////////////////////////////////////////

/*globals $,jQuery,document, window, navigator*/

/*!
 * ntp.video - Implements video playback
 */
 
 // Namespace
var ntp = ntp || {};

ntp.video = ntp.video || {};

(function () {
    var videoPlayer = null; // The MediaElementPlayer instance.
    var $videoPlayerLightbox = $('#video_player_lightbox');
    
    var currentId = null;
    
    $.extend(ntp.video, {
        
        videoWidth: 800,
        videoHeight: 450,
                
        play: function () {
            if (videoPlayer) {
                videoPlayer.load();
                videoPlayer.play();
            }
        },
        stop: function () {
            if (videoPlayer) {
                videoPlayer.pause();
                
                // Restore the status of video controls including the close button
                // so that the lightbox will be in the same state if it is shown again.
                ntp.video.controls.show();
            }
        },

        /**
         * Stops video player in other new tab page instances. It works in this way:
         * When a new video starts playing, a new ID is generated to represent the
         * current video, and passed to AutoCAD. AutoCAD then invokes ntp.video.stopWithJSON(idJson)
         * in all new tab page instances with the current video ID. If the ID is different
         * from the currentId stored in a new tab page instance, the video in that instance
         * is stopped.
         */
        stopOthers: function () {
            this.currentId = new Date().getTime().toString();
            ntp.acad.stopVideoById(this.currentId);
        },
        
        stopWithJSON: function (idJSON) {
            var json = $.parseJSON(idJSON);
            if (json.id !== this.currentId) {
                this.stop();
            }
        },

        updateViewedState: function () {
            var $videoPlayer = $('#video_player');
            if (ntp.content.learnTab.urlIsHilightedContent($videoPlayer.data('url'))) {
                ntp.acad.updateLearnContentState($videoPlayer.data('url'), $videoPlayer.data('upload_time'));
            }
        },

        showPlayer: function ($item) {
            // Initialize video player if needed
            if (videoPlayer === null) {
                var $video = $('#video_player');        
                if ($video.size() == 0) {
                    // Create video element
                    $video = $('<video>').attr({
                        id: 'video_player',
                        preload: 'none',
                        type: 'video/webm'
                    });
                    $('#video_container').append($video);
                }
                
                $('#video_container').width(this.videoWidth);
            
                $video.attr({
                    src: $item.data('url'),
                    width: this.videoWidth,
                    height: this.videoHeight
                });
            
                // Enable/disable automatic hiding player controls
                $video.on('play', 
                    ntp.utils.makeCallback(this.controls.autohide, 'enable'));
                $video.on('pause error ended',
                    ntp.utils.makeCallback(this.controls.autohide, 'disable'));
                
                // Stop video player in other new tab page instances
                $video.on('play', ntp.utils.makeCallback(this, 'stopOthers'));

                //Once Played mark it as a viewed item
                $video.on('play', ntp.utils.makeCallback(this, 'updateViewedState'));

                // Start playing video to play the video in autoplay mode
                videoPlayer = new MediaElementPlayer('#video_player', {
                    pluginWidth: this.videoWidth,
                    pluginHeight: this.videoHeight,
                    alwaysShowControls: true,
                    hideVolumeOnTouchDevices: false
                });
                
                $videoPlayerLightbox.lightbox('bind', 'lightboxshown', 
                    ntp.utils.makeCallback(this, 'play'));
            
                $videoPlayerLightbox.lightbox('bind', 'lightboxhidden',
                    ntp.utils.makeCallback(this, 'stop'));
            
                // Disable activating video control buttons with tab key
                $('#video_container button').attr('tabindex', '-1');
                // Required for 'click-to-pause' to work inside AutoCAD
                $('.mejs-mediaelement').click(ntp.utils.makeCallback(this, 'stop'));
            } else {
                // Update existing video player
                videoPlayer.pause();
                videoPlayer.setSrc($item.data('url'));
            }
        
            //Add some data to query in updateViewedState callback
            $('#video_player').data({
                url: $item.data('url'),
                upload_time: $item.data('upload_time')
            });

            // Update video player title and prev/next links
            $('#video_container .title').text($item.data('title'));
            var $prevLink = $('#video_container .prev');
            $prevLink.off('click');
            var $prevItem = $item.prev();
            if ($prevItem.size() > 0) {
                $prevLink.children('span').text($prevItem.data('title'));
                $prevLink.show();
                $prevLink.on('click', 
                    ntp.utils.makeCallbackWithArgs(this, 'showPlayer', $prevItem));
            } else {
                $prevLink.hide();
            }
        
            var $nextLink = $('#video_container .next');
            $nextLink.off('click');
            var $nextItem = $item.next();
            if ($nextItem.size() > 0) {
                $nextLink.children('span').text($nextItem.data('title'));
                $nextLink.show();
                $nextLink.on('click',
                    ntp.utils.makeCallbackWithArgs(this, 'showPlayer', $nextItem));
            } else {
                $nextLink.hide();
            }
        
            // Play video
            if ($videoPlayerLightbox.lightbox('isShowing')) {
                videoPlayer.load();
                videoPlayer.play();
            } else {
                $videoPlayerLightbox.lightbox();
            }
        }
    });
    
})();

/**
 * Manipulation of video player controls
 */
ntp.video.controls = (function () {
    var isVisible = true;
    var updateVisibility = function (visible) {
        if (visible !== isVisible) {
            if (visible) {
                $('#video_container .mejs-controls')
                    .add('#video_container header')
                    .add('#lightbox_panel .close')
                    .stop(true, true).css('opacity', 1); // Use opacity to avoid layout issues.
            } else {
                $('#video_container .mejs-controls')
                        .add('#video_container header')
                        .add('#lightbox_panel .close')
                        .stop(true, true).delay(1000).fadeTo(400, 0);
            }
            isVisible = visible;
        }
    };
    
    return {
        show: function () {
            updateVisibility(true);
        },
        hoverArea: function () {
            return $('#video_container')
                .add($('#lightbox_panel .close'));
        },
        hovering: function () {
            var isHovering = false;
            this.hoverArea().each(function () {
                isHovering = $(this).is(':hover');
                return !isHovering; // Break when any element is :hover
            });
            return isHovering;
        },
        refresh: function () {
            updateVisibility(this.hovering());
        }
    };    
})();

/**
 * Automatically hide video player controls when user is not hovering over the player
 * area.
 */
ntp.video.controls.autohide = (function () {
    var isEnabled = false;    
    return {
        enable: function () {
            if (!isEnabled) {
                if ($('#video_container .me-cannotplay').size() > 0) {
                    // Disallow enabling auto hide if video cannot be played
                    return;
                }
                ntp.video.controls.refresh();
                ntp.video.controls.hoverArea()
                    .on('mouseenter mouseleave', 
                        ntp.utils.makeCallback(ntp.video.controls, 'refresh'));
                isEnabled = true;
            }
        },
        disable: function () {
            if (isEnabled) {
                ntp.video.controls.hoverArea()
                    .off('mouseenter mouseleave');            
                ntp.video.controls.show();
                isEnabled = false;
            }
        }
    };
})();

// SIG // Begin signature block
// SIG // MIIZ4gYJKoZIhvcNAQcCoIIZ0zCCGc8CAQExDzANBglg
// SIG // hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
// SIG // BgEEAYI3AgEeMCQCAQEEEBDgyQbOONQRoqMAEEvTUJAC
// SIG // AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
// SIG // gSmWRtJg5qTgMwAnPK46dqGLWFLVV5CrZIUhN9/LOByg
// SIG // ggpSMIIFAzCCA+ugAwIBAgIQdnK0JhTS6n5kEFOlgziE
// SIG // ljANBgkqhkiG9w0BAQsFADCBhDELMAkGA1UEBhMCVVMx
// SIG // HTAbBgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8w
// SIG // HQYDVQQLExZTeW1hbnRlYyBUcnVzdCBOZXR3b3JrMTUw
// SIG // MwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1NiBD
// SIG // b2RlIFNpZ25pbmcgQ0EgLSBHMjAeFw0xNjA4MDgwMDAw
// SIG // MDBaFw0xNzA5MDIyMzU5NTlaMIGIMQswCQYDVQQGEwJV
// SIG // UzETMBEGA1UECAwKQ2FsaWZvcm5pYTETMBEGA1UEBwwK
// SIG // U2FuIFJhZmFlbDEWMBQGA1UECgwNQXV0b2Rlc2ssIElu
// SIG // YzEfMB0GA1UECwwWRGVzaWduIFNvbHV0aW9ucyBHcm91
// SIG // cDEWMBQGA1UEAwwNQXV0b2Rlc2ssIEluYzCCASIwDQYJ
// SIG // KoZIhvcNAQEBBQADggEPADCCAQoCggEBAOqZ9OjPzARq
// SIG // d9dP4fq2qXdpTJolFYJ8Wxg8WcZVciX3l+dljew4R9C8
// SIG // WR/0tW17lLANaSORF4IYmWTgdpFg5PGR3s+R4A461hpR
// SIG // IG9QthGUq8Bt7af5VbCjjKtYBNq/x3ukqVsw/1/qljGV
// SIG // plDOyr976ktnCm0/wL0N+ubk5WAIgv+I9E0i6+GX1hRm
// SIG // +eRliYKeKF0/gBLyDh3ut+N3HNPrnfjfec7q2Czd82Ce
// SIG // RBCToR1E7Cp6yBBdGUfb3JE0PnE1+6n0qJnsvs2bsCIz
// SIG // qVo3A0iMHBsbioxa5DYyXuEfNCktosUlFXfegd514ZJn
// SIG // s6YDr4PlahC0+lK3hgJy5vkCAwEAAaOCAWkwggFlMAkG
// SIG // A1UdEwQCMAAwDgYDVR0PAQH/BAQDAgeAMBMGA1UdJQQM
// SIG // MAoGCCsGAQUFBwMDMGEGA1UdIARaMFgwVgYGZ4EMAQQB
// SIG // MEwwIwYIKwYBBQUHAgEWF2h0dHBzOi8vZC5zeW1jYi5j
// SIG // b20vY3BzMCUGCCsGAQUFBwICMBkMF2h0dHBzOi8vZC5z
// SIG // eW1jYi5jb20vcnBhMB8GA1UdIwQYMBaAFNTABiJJ6zlL
// SIG // 3ZPiXKG4R3YJcgNYMCsGA1UdHwQkMCIwIKAeoByGGmh0
// SIG // dHA6Ly9yYi5zeW1jYi5jb20vcmIuY3JsMFcGCCsGAQUF
// SIG // BwEBBEswSTAfBggrBgEFBQcwAYYTaHR0cDovL3JiLnN5
// SIG // bWNkLmNvbTAmBggrBgEFBQcwAoYaaHR0cDovL3JiLnN5
// SIG // bWNiLmNvbS9yYi5jcnQwEQYJYIZIAYb4QgEBBAQDAgQQ
// SIG // MBYGCisGAQQBgjcCARsECDAGAQEAAQH/MA0GCSqGSIb3
// SIG // DQEBCwUAA4IBAQDAO29k596Wq5ympOToYLmaRc7ZeGvY
// SIG // x6j5lNWmwCC9ACLTN+mqXF1msf5KtHRHTpyLFVDH/zCC
// SIG // 3LrRzHSHuflhkKFlhJQsEttZ3rhmKxlEtJ85Id9pA8wm
// SIG // XN+Q5tKIStReWLsZ2eA15G9BEFmPvq5DAFj0h+LNF6hq
// SIG // 87C2bVqim29Kf9wDMp3Ndd7hj07QEVh7CqCKEoAJYvXR
// SIG // BBAdrMTqjCTtYCwDkAaAg5LdEm5w76jCQkR2XzKOTgDl
// SIG // qa9uQIyQBAc2ci6X9OBdGJw0ZM0JDlEkRpr3uODtZIOn
// SIG // UQQchI+k+cwEAK7+vFhfsLBi7Dyt1mbvfINr803cFja3
// SIG // i5JOMIIFRzCCBC+gAwIBAgIQfBs1NUrn23TnQV8Racpr
// SIG // qDANBgkqhkiG9w0BAQsFADCBvTELMAkGA1UEBhMCVVMx
// SIG // FzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQL
// SIG // ExZWZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTowOAYDVQQL
// SIG // EzEoYykgMjAwOCBWZXJpU2lnbiwgSW5jLiAtIEZvciBh
// SIG // dXRob3JpemVkIHVzZSBvbmx5MTgwNgYDVQQDEy9WZXJp
// SIG // U2lnbiBVbml2ZXJzYWwgUm9vdCBDZXJ0aWZpY2F0aW9u
// SIG // IEF1dGhvcml0eTAeFw0xNDA3MjIwMDAwMDBaFw0yNDA3
// SIG // MjEyMzU5NTlaMIGEMQswCQYDVQQGEwJVUzEdMBsGA1UE
// SIG // ChMUU3ltYW50ZWMgQ29ycG9yYXRpb24xHzAdBgNVBAsT
// SIG // FlN5bWFudGVjIFRydXN0IE5ldHdvcmsxNTAzBgNVBAMT
// SIG // LFN5bWFudGVjIENsYXNzIDMgU0hBMjU2IENvZGUgU2ln
// SIG // bmluZyBDQSAtIEcyMIIBIjANBgkqhkiG9w0BAQEFAAOC
// SIG // AQ8AMIIBCgKCAQEA15VD1NzfZ645+1KktiYxBHDpt45b
// SIG // Kro3aTWVj7vAMOeG2HO73+vRdj+KVo7rLUvwVxhOsY2l
// SIG // M9MLdSPVankn3aPT9w6HZbXerRzx9TW0IlGvIqHBXUuQ
// SIG // f8BZTqudeakC1x5JsTtNh/7CeKu/71KunK8I2TnlmlE+
// SIG // aV8wEE5xY2xY4fAgMxsPdL5byxLh24zEgJRyu/ZFmp7B
// SIG // JQv7oxye2KYJcHHswEdMj33D3hnOPu4Eco4X0//wsgUy
// SIG // GUzTsByf/qV4IEJwQbAmjG8AyDoAEUF6QbCnipEEoJl4
// SIG // 9He082Aq5mxQBLcUYP8NUfSoi4T+IdpcXn31KXlPsER0
// SIG // b21y/wIDAQABo4IBeDCCAXQwLgYIKwYBBQUHAQEEIjAg
// SIG // MB4GCCsGAQUFBzABhhJodHRwOi8vcy5zeW1jZC5jb20w
// SIG // EgYDVR0TAQH/BAgwBgEB/wIBADBmBgNVHSAEXzBdMFsG
// SIG // C2CGSAGG+EUBBxcDMEwwIwYIKwYBBQUHAgEWF2h0dHBz
// SIG // Oi8vZC5zeW1jYi5jb20vY3BzMCUGCCsGAQUFBwICMBka
// SIG // F2h0dHBzOi8vZC5zeW1jYi5jb20vcnBhMDYGA1UdHwQv
// SIG // MC0wK6ApoCeGJWh0dHA6Ly9zLnN5bWNiLmNvbS91bml2
// SIG // ZXJzYWwtcm9vdC5jcmwwEwYDVR0lBAwwCgYIKwYBBQUH
// SIG // AwMwDgYDVR0PAQH/BAQDAgEGMCkGA1UdEQQiMCCkHjAc
// SIG // MRowGAYDVQQDExFTeW1hbnRlY1BLSS0xLTcyNDAdBgNV
// SIG // HQ4EFgQU1MAGIknrOUvdk+JcobhHdglyA1gwHwYDVR0j
// SIG // BBgwFoAUtnf6aUhHn1MS1cLqBzJ2B9GXBxkwDQYJKoZI
// SIG // hvcNAQELBQADggEBAH/ryqfqi3ZC6z6OIFQw47e53PpI
// SIG // PhbHD0WVEM0nhqNm8wLtcfiqwlWXkXCD+VJ+Umk8yfHg
// SIG // lEaAGLuh1KRWpvMdAJHVhvNIh+DLxDRoIF60y/kF7Zyv
// SIG // cFMnueg+flGgaXGL3FHtgDolMp9Er25DKNMhdbuX2IuL
// SIG // jP6pBEYEhfcVnEsRjcQsF/7Vbn+a4laS8ZazrS359N/a
// SIG // iZnOsjhEwPdHe8olufoqaDObUHLeqJ/UzSwLNL2LMHhA
// SIG // 4I2OJxuQbxq+CBWBXesv4lHnUR7JeCnnHmW/OO8BSgEJ
// SIG // JA4WxBR5wUE3NNA9kVKUneFo7wjw4mmcZ26QCxqTcdQm
// SIG // AsPAWiMxgg7oMIIO5AIBATCBmTCBhDELMAkGA1UEBhMC
// SIG // VVMxHTAbBgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9u
// SIG // MR8wHQYDVQQLExZTeW1hbnRlYyBUcnVzdCBOZXR3b3Jr
// SIG // MTUwMwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1
// SIG // NiBDb2RlIFNpZ25pbmcgQ0EgLSBHMgIQdnK0JhTS6n5k
// SIG // EFOlgziEljANBglghkgBZQMEAgEFAKB8MBAGCisGAQQB
// SIG // gjcCAQwxAjAAMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3
// SIG // AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEV
// SIG // MC8GCSqGSIb3DQEJBDEiBCC+XAHIPvciE/oUsyYVSPWm
// SIG // SH0GYblyL+7NpItU5aadoTANBgkqhkiG9w0BAQEFAASC
// SIG // AQAzJWTqC9NlIVnEAMv1oHtcx2+c+XSsjqJGQMMSNyCf
// SIG // RW2Afaaj9pWb2eggdBnyYStj6EmXaYFvKJBYGzOGjAX2
// SIG // ka9XQtDTRGslxLZUIgj1ndAF3nnzETykyhoxBvftNLzk
// SIG // aRtssX2G6vRGedofNZuCSzIhIn5C2EjV11pYR9GwWzLK
// SIG // v8xJxkE7Lwvfjn7vlcjICvrlKicMzGsMiUYPvJddJrKz
// SIG // HnZ9XFK7l5gmrhg/8PXmqCw/yVb4P992KLeYEl604FAm
// SIG // Vg0Hu+WpPmD/pPSGofIrMJmNS1APh2cHDwFoTIepIUr7
// SIG // NO0frSeKZpVSimIYOb8mMkO9xGjsbnjIQDwsoYIMoTCC
// SIG // DJ0GCisGAQQBgjcDAwExggyNMIIMiQYJKoZIhvcNAQcC
// SIG // oIIMejCCDHYCAQMxDzANBglghkgBZQMEAgEFADCB3QYL
// SIG // KoZIhvcNAQkQAQSggc0EgcowgccCAQEGCSsGAQQBoDIC
// SIG // AzAxMA0GCWCGSAFlAwQCAQUABCC4+FVnaiLfbNQ80Lg+
// SIG // D5f89/ARHsA35C05JZ8wANqzfwIUa7wEp89uQyt/Ndjo
// SIG // +uNCdwxxdKoYDzIwMTcwMjAzMDQ1MDM3WqBdpFswWTEL
// SIG // MAkGA1UEBhMCU0cxHzAdBgNVBAoTFkdNTyBHbG9iYWxT
// SIG // aWduIFB0ZSBMdGQxKTAnBgNVBAMTIEdsb2JhbFNpZ24g
// SIG // VFNBIGZvciBBZHZhbmNlZCAtIEcyoIIIxjCCBKkwggOR
// SIG // oAMCAQICEhEhBvEPzmjwm/rlWxjNjyABdzANBgkqhkiG
// SIG // 9w0BAQsFADBbMQswCQYDVQQGEwJCRTEZMBcGA1UEChMQ
// SIG // R2xvYmFsU2lnbiBudi1zYTExMC8GA1UEAxMoR2xvYmFs
// SIG // U2lnbiBUaW1lc3RhbXBpbmcgQ0EgLSBTSEEyNTYgLSBH
// SIG // MjAeFw0xNjA1MjQwMDAwMDBaFw0yNzA2MjQwMDAwMDBa
// SIG // MFkxCzAJBgNVBAYTAlNHMR8wHQYDVQQKExZHTU8gR2xv
// SIG // YmFsU2lnbiBQdGUgTHRkMSkwJwYDVQQDEyBHbG9iYWxT
// SIG // aWduIFRTQSBmb3IgQWR2YW5jZWQgLSBHMjCCASIwDQYJ
// SIG // KoZIhvcNAQEBBQADggEPADCCAQoCggEBALfHkooo2POR
// SIG // y1ANXespRMGCWaXKZM69g7VR5ZTMboCaF2zc/2LmNkNe
// SIG // AcIMZI3Kd572XXdFuV7IJOtBNxFmN6zIzXSbzLPvTOJ/
// SIG // G85zvsmWnTUefPdU92zsoBLWrpmdY8R4X1mpLiL1wyfY
// SIG // sltFYyeQ/4yxPam08w7A8SBlBomdAxyjsFJBhTTrvMvO
// SIG // VPYS/rMBiUqm+lTFH/vTHMDjv5fjP9Ab+UDHG9XrJnxD
// SIG // MMdw8ngRqoVOpQ4NAEo6EXejyiMBgJ7Ik1ZdRsyK2NKq
// SIG // CoSFsolb1TLOQXsYTlTKq9FSXhLTJJ5W8wyP3b2SjnnV
// SIG // QYnDo6DlkfzHZ52HM85xMnMCAwEAAaOCAWcwggFjMA4G
// SIG // A1UdDwEB/wQEAwIHgDBMBgNVHSAERTBDMEEGCSsGAQQB
// SIG // oDIBHjA0MDIGCCsGAQUFBwIBFiZodHRwczovL3d3dy5n
// SIG // bG9iYWxzaWduLmNvbS9yZXBvc2l0b3J5LzAJBgNVHRME
// SIG // AjAAMBYGA1UdJQEB/wQMMAoGCCsGAQUFBwMIMEYGA1Ud
// SIG // HwQ/MD0wO6A5oDeGNWh0dHA6Ly9jcmwuZ2xvYmFsc2ln
// SIG // bi5jb20vZ3MvZ3N0aW1lc3RhbXBpbmdzaGEyZzIuY3Js
// SIG // MFgGCCsGAQUFBwEBBEwwSjBIBggrBgEFBQcwAoY8aHR0
// SIG // cDovL3NlY3VyZS5nbG9iYWxzaWduLmNvbS9jYWNlcnQv
// SIG // Z3N0aW1lc3RhbXBpbmdzaGEyZzIuY3J0MB0GA1UdDgQW
// SIG // BBQtbm7RjeUDgO7nY+mn2doLPFciPTAfBgNVHSMEGDAW
// SIG // gBSSIadKlV1ksJu0HuYAN0fmnUErTDANBgkqhkiG9w0B
// SIG // AQsFAAOCAQEAV51T5N3upSze5L9igKJhhkqfm50kIzCb
// SIG // jyeHL/oEWc5wiD1GUnfEm0XSj723IRhJ2C6H/5Iud/k/
// SIG // CvmgIVwTT+SEKyiHzFwVuROr4hJVw/hFHkkZzqp1DyHo
// SIG // 71H8NCwLMgUJsuQWaa3ZLn7h/C1IvxrTdDUBOt8wQ3Bn
// SIG // ejjXuhHCVvsxSLpb8SESYuB2iZEfSTjUWE15CYqp2m8C
// SIG // 1q3k2ol9TNmxMHBAattFulN2kNxLQhYhz+TSWJTUVWWb
// SIG // dgOsrhgItoMSjEE+X4BFZMiJ1DMXoaFMvT/Ekv5/hfK+
// SIG // sazX9p7LzhMq7gJDT/z/cDU0ozN8z+INMiQgfAA+ozIR
// SIG // 3jCCBBUwggL9oAMCAQICCwQAAAAAATGJxlAEMA0GCSqG
// SIG // SIb3DQEBCwUAMEwxIDAeBgNVBAsTF0dsb2JhbFNpZ24g
// SIG // Um9vdCBDQSAtIFIzMRMwEQYDVQQKEwpHbG9iYWxTaWdu
// SIG // MRMwEQYDVQQDEwpHbG9iYWxTaWduMB4XDTExMDgwMjEw
// SIG // MDAwMFoXDTI5MDMyOTEwMDAwMFowWzELMAkGA1UEBhMC
// SIG // QkUxGTAXBgNVBAoTEEdsb2JhbFNpZ24gbnYtc2ExMTAv
// SIG // BgNVBAMTKEdsb2JhbFNpZ24gVGltZXN0YW1waW5nIENB
// SIG // IC0gU0hBMjU2IC0gRzIwggEiMA0GCSqGSIb3DQEBAQUA
// SIG // A4IBDwAwggEKAoIBAQCqm47DqxFRJQG2lpTiT9jBCPZG
// SIG // I9lFxZWXW6sav9JsV8kzBh+gD8Y8flNIer+dh56v7sOM
// SIG // R+FC7OPjoUpsDBfEpsG5zVvxHkSJjv4L3iFYE+5NyMVn
// SIG // Cxyys/E0dpGiywdtN8WgRyYCFaSQkal5ntfrV50rfCLY
// SIG // FNfxBx54IjZrd3mvr/l/jk7htQgx/ertS3FijCPxAzmP
// SIG // RHm2dgNXnq0vCEbc0oy89I50zshoaVF2EYsPXSRbGVQ9
// SIG // JsxAjYInG1kgfVn2k4CO+Co4/WugQGUfV3bMW44ETyyo
// SIG // 24RQE0/G3Iu5+N1pTIjrnHswJvx6WLtZvBRykoFXt3bJ
// SIG // 2IAKgG4JAgMBAAGjgegwgeUwDgYDVR0PAQH/BAQDAgEG
// SIG // MBIGA1UdEwEB/wQIMAYBAf8CAQAwHQYDVR0OBBYEFJIh
// SIG // p0qVXWSwm7Qe5gA3R+adQStMMEcGA1UdIARAMD4wPAYE
// SIG // VR0gADA0MDIGCCsGAQUFBwIBFiZodHRwczovL3d3dy5n
// SIG // bG9iYWxzaWduLmNvbS9yZXBvc2l0b3J5LzA2BgNVHR8E
// SIG // LzAtMCugKaAnhiVodHRwOi8vY3JsLmdsb2JhbHNpZ24u
// SIG // bmV0L3Jvb3QtcjMuY3JsMB8GA1UdIwQYMBaAFI/wS3+o
// SIG // LkUkrk1Q+mOai97i3Ru8MA0GCSqGSIb3DQEBCwUAA4IB
// SIG // AQAEVoJKfNDOyb82ZtG+NZ6TbJfoBs4xGFn5bEFfgC7A
// SIG // QiW4GMf81LE3xGigzyhqA3RLY5eFd2E71y/j9b0zopJ9
// SIG // ER+eimzvLLD0Yo02c9EWNvG8Xuy0gJh4/NJ2eejhIZTg
// SIG // H8Si4apn27Occ+VAIs85ztvmd5Wnu7LL9hmGnZ/I1JgF
// SIG // snFvTnWu8T1kajteTkamKl0IkvGj8x10v2INI4xcKjiV
// SIG // 0sDVzc+I2h8otbqBaWQqtaai1XOv3EbbBK6R127FmLrU
// SIG // R8RWdIBHeFiMvu8r/exsv9GU979Q4HvgkP0gGHgYIl0I
// SIG // LowcoJfzHZl9o52R0wZETgRuehwg4zbwtlC5MYICtDCC
// SIG // ArACAQEwcTBbMQswCQYDVQQGEwJCRTEZMBcGA1UEChMQ
// SIG // R2xvYmFsU2lnbiBudi1zYTExMC8GA1UEAxMoR2xvYmFs
// SIG // U2lnbiBUaW1lc3RhbXBpbmcgQ0EgLSBTSEEyNTYgLSBH
// SIG // MgISESEG8Q/OaPCb+uVbGM2PIAF3MA0GCWCGSAFlAwQC
// SIG // AQUAoIIBFDAaBgkqhkiG9w0BCQMxDQYLKoZIhvcNAQkQ
// SIG // AQQwHAYJKoZIhvcNAQkFMQ8XDTE3MDIwMzA0NTAzN1ow
// SIG // LwYJKoZIhvcNAQkEMSIEIERPAdx3cG6xxgAv0IuwAt3F
// SIG // M447LUX5vhyUmbgjQQYhMIGmBgsqhkiG9w0BCRACDDGB
// SIG // ljCBkzCBkDCBjQQUfVXY51pWovxzgkP3uFSHXFy1Kg0w
// SIG // dTBfpF0wWzELMAkGA1UEBhMCQkUxGTAXBgNVBAoTEEds
// SIG // b2JhbFNpZ24gbnYtc2ExMTAvBgNVBAMTKEdsb2JhbFNp
// SIG // Z24gVGltZXN0YW1waW5nIENBIC0gU0hBMjU2IC0gRzIC
// SIG // EhEhBvEPzmjwm/rlWxjNjyABdzANBgkqhkiG9w0BAQEF
// SIG // AASCAQAL8OPQBqrfOSpUDo4nyn1cvadsFst0TBTGmPuf
// SIG // skRN5pRpmQ3VjLVBlYpKMrKQ6iSPs2Uep4tish/8dzfr
// SIG // 6MCj/T7X604IFc/73/sP4aUaX0GCjJtdokq4A37s+PTM
// SIG // QnFMFUSdWzD24IyMH0FQx1n50mM0vKc0eba7loOP0QIC
// SIG // pZAyhPg0IbzZAMXIMd/oPcfLxLZcyaebn5DgmVnfQLlq
// SIG // clEac2DogU/oRR/hpyH2m4gkFwobH2+eyjQXtxKGR80k
// SIG // QOokhCW1B9AJyhnj9sRHEMQNj4aUx3TmzlvaEtK7180J
// SIG // b3A3PT9woMqjX/v4+GDp/yvIp3pqkrsVK3DxPcg+
// SIG // End signature block
